<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Core\Helpers;
use App\Core\Telegram;
use App\Repositories\ForcedChannelRepo;
use App\Repositories\UserRepo;
use App\Services\AuthService;

final class AdminController
{
    private UserRepo $users;
    private ForcedChannelRepo $channels;

    public function __construct(private Telegram $tg, private AuthService $auth)
    {
        $this->users = new UserRepo();
        $this->channels = new ForcedChannelRepo();
    }

    public function panel(int $chatId, int $userId): void
    {
        if (!$this->auth->isAdmin($userId)) { $this->tg->sendMessage($chatId, "غير مسموح."); return; }

        $kb = [
            [[ 'text'=>'➕ تعيين أدمن', 'callback_data'=>'admin:setadmin' ], [ 'text'=>'🗑️ حظر مستخدم', 'callback_data'=>'admin:ban' ]],
            [[ 'text'=>'📊 إحصائيات', 'callback_data'=>'admin:stats' ], [ 'text'=>'📢 إذاعة', 'callback_data'=>'admin:broadcast' ]],
            [[ 'text'=>'📌 القنوات الإجبارية', 'callback_data'=>'admin:channels' ]],
        ];
        $this->tg->sendMessage($chatId, "👑 <b>لوحة الأدمن</b>\n\nاختر خياراً:", ['reply_markup'=>json_encode(['inline_keyboard'=>$kb], JSON_UNESCAPED_UNICODE)]);
    }

    public function handleState(int $chatId, int $userId, string $text, array $stateRow): void
    {
        if (!$this->auth->isAdmin($userId)) return;
        $state = (string)$stateRow['state'];
        $clean = Helpers::cleanText($text);

        if ($state === 'admin:setadmin') {
            $cid = (int)preg_replace('/\D+/', '', $clean);
            $ok = $cid ? $this->users->setRoleByChatId($cid, 'ADMIN') : false;
            $this->auth->clearState($userId);
            $this->tg->sendMessage($chatId, $ok ? "✅ تم تعيينه أدمن" : "❌ chat_id غير صحيح أو المستخدم غير موجود");
            return;
        }
        if ($state === 'admin:ban') {
            $cid = (int)preg_replace('/\D+/', '', $clean);
            $ok = $cid ? $this->users->banByChatId($cid) : false;
            $this->auth->clearState($userId);
            $this->tg->sendMessage($chatId, $ok ? "✅ تم حظر المستخدم" : "❌ chat_id غير صحيح أو المستخدم غير موجود");
            return;
        }
        if ($state === 'admin:broadcast') {
            $this->auth->clearState($userId);
            $sent = $this->users->broadcast($this->tg, $clean);
            $this->tg->sendMessage($chatId, "✅ تم إرسال الإذاعة إلى {$sent} مستخدم.");
            return;
        }
        if ($state === 'admin:add_channel') {
            $lines = preg_split('/\R/u', $clean);
            $username = trim($lines[0] ?? '');
            $invite = trim($lines[1] ?? '');
            if ($username && $username[0] !== '@') $username='@'.$username;
            if ($username==='' || $username==='@') { $this->tg->sendMessage($chatId, "اكتب @username صحيح"); return; }
            if ($invite==='') $invite='https://t.me/'.ltrim($username,'@');
            $this->channels->add($username, $invite);
            $this->auth->clearState($userId);
            $this->tg->sendMessage($chatId, "✅ تمت إضافة القناة.");
            $this->showChannels($chatId);
            return;
        }
    }

    public function handleCallback(int $chatId, int $userId, string $cbId, string $data): void
    {
        if (!$this->auth->isAdmin($userId)) { $this->tg->answerCallbackQuery($cbId, "غير مسموح"); return; }

        if ($data === 'admin:stats') {
            $u=$this->users->countUsers(); $a=$this->users->countAdmins(); $b=$this->users->countBanned(); $c=$this->channels->countActive();
            $this->tg->answerCallbackQuery($cbId,'');
            $this->tg->sendMessage($chatId, "📊 <b>إحصائيات</b>\n\n👥 المستخدمون: {$u}\n👑 الأدمن: {$a}\n🚫 محظورون: {$b}\n📌 قنوات إجبارية: {$c}");
            return;
        }
        if ($data === 'admin:setadmin') { $this->auth->setState($userId,'admin:setadmin',[]); $this->tg->answerCallbackQuery($cbId,''); $this->tg->sendMessage($chatId,"أرسل chat_id للمستخدم:"); return; }
        if ($data === 'admin:ban') { $this->auth->setState($userId,'admin:ban',[]); $this->tg->answerCallbackQuery($cbId,''); $this->tg->sendMessage($chatId,"أرسل chat_id للحظر:"); return; }
        if ($data === 'admin:broadcast') { $this->auth->setState($userId,'admin:broadcast',[]); $this->tg->answerCallbackQuery($cbId,''); $this->tg->sendMessage($chatId,"أرسل نص الإذاعة:"); return; }
        if ($data === 'admin:channels') { $this->tg->answerCallbackQuery($cbId,''); $this->showChannels($chatId); return; }
        if ($data === 'admin:channels:add') { $this->auth->setState($userId,'admin:add_channel',[]); $this->tg->answerCallbackQuery($cbId,''); $this->tg->sendMessage($chatId,"أرسل @username ثم رابط (اختياري) بسطر ثاني:"); return; }
        if (str_starts_with($data,'admin:channels:del:')) { $id=(int)substr($data, strlen('admin:channels:del:')); $this->channels->deactivate($id); $this->tg->answerCallbackQuery($cbId,'✅'); $this->showChannels($chatId); return; }

        $this->tg->answerCallbackQuery($cbId,'');
    }

    private function showChannels(int $chatId): void
    {
        $rows = $this->channels->listActive();
        $text = "📌 <b>القنوات الإجبارية</b>\n\n";
        $kb = [];
        $kb[] = [[ 'text'=>'➕ إضافة قناة', 'callback_data'=>'admin:channels:add' ]];
        if (!$rows) $text .= "لا توجد قنوات.";
        else {
            foreach ($rows as $r) {
                $text .= "#{$r['id']} {$r['channel_username']}\n";
                $kb[] = [[ 'text'=>'🗑️ حذف ' . $r['channel_username'], 'callback_data'=>'admin:channels:del:' . $r['id'] ]];
            }
        }
        $this->tg->sendMessage($chatId, $text, ['reply_markup'=>json_encode(['inline_keyboard'=>$kb], JSON_UNESCAPED_UNICODE)]);
    }
}
