<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Core\Helpers;
use App\Core\Telegram;
use App\Repositories\TaskRepo;
use App\Repositories\TeamRepo;
use App\Services\AuthService;

final class TaskController
{
    private TaskRepo $tasks;
    private TeamRepo $teams;

    public function __construct(private Telegram $tg, private AuthService $auth)
    {
        $this->tasks = new TaskRepo();
        $this->teams = new TeamRepo();
    }

    public function showMyTasks(int $chatId, int $userId): void
    {
        $personal = $this->tasks->listPersonal($userId, 50);
        $assigned = $this->tasks->listAssignedToUser($userId, 50);

        $text = "✅ <b>مهامي</b>\n\n";
        $kb = [];

        $text .= "📝 <b>مهامي الشخصية</b>:\n";
        if (!$personal) $text .= "- لا توجد مهام.\n";
        else {
            foreach ($personal as $t) {
                $label = "#{$t['id']} " . Helpers::statusLabel($t['status']) . " - " . Helpers::esc(mb_strimwidth((string)$t['title'], 0, 40, '…','UTF-8'));
                $kb[] = [[ 'text' => $label, 'callback_data' => 'task:toggle:' . $t['id'] ]];
            }
        }

        $text .= "\n👥 <b>مهام مسندة إليّ (فرق)</b>:\n";
        if (!$assigned) $text .= "- لا توجد مهام.\n";
        else {
            foreach ($assigned as $t) {
                $team = $t['team_name'] ? ("[" . Helpers::esc($t['team_name']) . "] ") : "";
                $label = $team . "#{$t['id']} " . Helpers::statusLabel($t['status']) . " - " . Helpers::esc(mb_strimwidth((string)$t['title'], 0, 35, '…','UTF-8'));
                $kb[] = [[ 'text' => $label, 'callback_data' => 'task:toggle:' . $t['id'] ]];
            }
        }

        $kb[] = [[ 'text'=>'➕ مهمة شخصية', 'callback_data'=>'task:add_personal' ], [ 'text'=>'➕ مهمة لفريق', 'callback_data'=>'task:add_team' ]];

        $this->tg->sendMessage($chatId, $text, [
            'reply_markup' => json_encode(['inline_keyboard'=>$kb], JSON_UNESCAPED_UNICODE)
        ]);
    }

    public function askAddPersonal(int $chatId, int $userId): void
    {
        $this->auth->setState($userId, 'task:add_personal', []);
        $this->tg->sendMessage($chatId, "✍️ ارسل عنوان المهمة الشخصية:");
    }

    public function handleState(int $chatId, int $userId, string $text, array $stateRow): void
    {
        $state = (string)$stateRow['state'];
        $payload = $stateRow['payload'] ?? [];

        if ($state === 'task:add_personal') {
            $title = Helpers::cleanText($text);
            if ($title === '') { $this->tg->sendMessage($chatId, "اكتب عنوان صحيح."); return; }
            $this->tasks->createPersonal($userId, $title);
            $this->auth->clearState($userId);
            $this->tg->sendMessage($chatId, "✅ تم إضافة المهمة.");
            $this->showMyTasks($chatId, $userId);
            return;
        }

        if ($state === 'task:add_team:ask_title') {
            $title = Helpers::cleanText($text);
            if ($title === '') { $this->tg->sendMessage($chatId, "اكتب عنوان صحيح."); return; }
            $payload['title'] = $title;
            $this->auth->setState($userId, 'task:add_team:pick_assignee', $payload);
            $this->promptPickAssignee($chatId, (int)$payload['team_id']);
            return;
        }
    }

    public function handleCallback(int $chatId, int $userId, string $cbId, string $data, array $msg): void
    {
        if ($data === 'task:add_personal') {
            $this->tg->answerCallbackQuery($cbId, '');
            $this->askAddPersonal($chatId, $userId);
            return;
        }

        if ($data === 'task:add_team') {
            $this->tg->answerCallbackQuery($cbId, '');
            $this->promptPickTeamForTask($chatId, $userId);
            return;
        }

        if (str_starts_with($data, 'task:pickteam:')) {
            $teamId = (int)substr($data, strlen('task:pickteam:'));
            if (!$this->teams->isMember($teamId, $userId)) { $this->tg->answerCallbackQuery($cbId, 'ليس عضو'); return; }
            $this->auth->setState($userId, 'task:add_team:ask_title', ['team_id'=>$teamId]);
            $this->tg->answerCallbackQuery($cbId, '');
            $this->tg->sendMessage($chatId, "✍️ ارسل عنوان المهمة للفريق:");
            return;
        }

        if (str_starts_with($data, 'task:pickassignee:')) {
            $assignee = substr($data, strlen('task:pickassignee:'));
            $st = $this->auth->getStateRow($userId);
            $payload = $st['payload'] ?? [];
            $teamId = (int)($payload['team_id'] ?? 0);
            $title = (string)($payload['title'] ?? '');
            if (!$teamId || $title === '') { $this->tg->answerCallbackQuery($cbId, 'انتهت'); $this->auth->clearState($userId); return; }

            $assignedUserId = ($assignee === 'none') ? null : (int)$assignee;
            $this->tasks->createTeamTask($teamId, $userId, $assignedUserId, $title);
            $this->auth->clearState($userId);
            $this->tg->answerCallbackQuery($cbId, '✅ تم');
            $this->showMyTasks($chatId, $userId);
            return;
        }

        if (str_starts_with($data, 'task:toggle:')) {
            $taskId = (int)substr($data, strlen('task:toggle:'));
            $task = $this->tasks->getById($taskId);
            if (!$task) { $this->tg->answerCallbackQuery($cbId, 'غير موجود'); return; }

            $allowed = false;
            if ($task['team_id'] === null) $allowed = ((int)$task['owner_user_id'] === $userId);
            else $allowed = ((int)$task['assigned_user_id'] === $userId) || $this->teams->isOwner((int)$task['team_id'], $userId);

            if (!$allowed) { $this->tg->answerCallbackQuery($cbId, 'لا صلاحية'); return; }

            $new = Helpers::nextStatus((string)$task['status']);
            $this->tasks->updateStatus($taskId, $new);
            $this->tg->answerCallbackQuery($cbId, Helpers::statusLabel($new));
            $this->showMyTasks($chatId, $userId);
            return;
        }

        $this->tg->answerCallbackQuery($cbId, '');
    }

    private function promptPickTeamForTask(int $chatId, int $userId): void
    {
        $teams = $this->teams->listForUser($userId);
        if (!$teams) { $this->tg->sendMessage($chatId, "لا توجد فرق لديك."); return; }
        $kb = [];
        foreach ($teams as $t) $kb[] = [[ 'text' => '👥 ' . Helpers::esc($t['name']), 'callback_data' => 'task:pickteam:' . $t['id'] ]];
        $this->tg->sendMessage($chatId, "اختر الفريق:", ['reply_markup'=>json_encode(['inline_keyboard'=>$kb], JSON_UNESCAPED_UNICODE)]);
    }

    private function promptPickAssignee(int $chatId, int $teamId): void
    {
        $members = $this->teams->listMembers($teamId);
        $kb = [];
        $kb[] = [[ 'text' => '🚫 بدون إسناد', 'callback_data' => 'task:pickassignee:none' ]];
        foreach ($members as $m) {
            $name = trim(($m['first_name'] ?? '') . ' ' . ($m['last_name'] ?? ''));
            if ($name === '') $name = '@' . ($m['username'] ?? 'user');
            $kb[] = [[ 'text' => '👤 ' . Helpers::esc($name) . ' (' . $m['member_role'] . ')', 'callback_data' => 'task:pickassignee:' . $m['user_id'] ]];
        }
        $this->tg->sendMessage($chatId, "اختر الشخص المسند له:", ['reply_markup'=>json_encode(['inline_keyboard'=>$kb], JSON_UNESCAPED_UNICODE)]);
    }
}
