<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Core\Config;
use App\Core\Helpers;
use App\Core\Telegram;
use App\Repositories\TeamRepo;
use App\Services\AuthService;

final class TeamController
{
    private TeamRepo $teams;

    public function __construct(private Telegram $tg, private AuthService $auth)
    {
        $this->teams = new TeamRepo();
    }

    public function showTeams(int $chatId, int $userId): void
    {
        $rows = $this->teams->listForUser($userId);
        $kb = [];
        $kb[] = [[ 'text' => '➕ إنشاء فريق', 'callback_data' => 'team:create' ]];
        foreach ($rows as $t) $kb[] = [[ 'text' => '👥 ' . Helpers::esc($t['name']), 'callback_data' => 'team:view:' . $t['id'] ]];

        $text = "👥 <b>فرقي</b>\n\n" . ($rows ? "اختر فريقاً:" : "لا توجد فرق.");
        $this->tg->sendMessage($chatId, $text, ['reply_markup'=>json_encode(['inline_keyboard'=>$kb], JSON_UNESCAPED_UNICODE)]);
    }

    public function handleJoin(int $chatId, int $userId, string $token): void
    {
        $team = $this->teams->getByInviteToken($token);
        if (!$team) { $this->tg->sendMessage($chatId, "❌ رابط الدعوة غير صالح."); return; }
        $this->teams->addMember((int)$team['id'], $userId, 'MEMBER');
        $this->tg->sendMessage($chatId, "✅ تم الانضمام إلى الفريق: <b>" . Helpers::esc($team['name']) . "</b>");
        $this->showTeams($chatId, $userId);
    }

    public function handleState(int $chatId, int $userId, string $text, array $stateRow): void
    {
        if ((string)$stateRow['state'] !== 'team:create') return;
        $name = Helpers::cleanText($text);
        if ($name === '') { $this->tg->sendMessage($chatId, "اكتب اسم صحيح."); return; }
        $token = Helpers::randToken(18);
        $teamId = $this->teams->create($userId, $name, $token);
        $this->teams->addMember($teamId, $userId, 'OWNER');
        $this->auth->clearState($userId);

        $link = "https://t.me/" . Config::BOT_USERNAME . "?start=join_" . $token;
        $this->tg->sendMessage($chatId, "✅ تم إنشاء الفريق <b>" . Helpers::esc($name) . "</b>\n\nرابط الدعوة:\n" . Helpers::esc($link));
        $this->showTeams($chatId, $userId);
    }

    public function handleCallback(int $chatId, int $userId, string $cbId, string $data, array $msg): void
    {
        if ($data === 'team:create') {
            $this->auth->setState($userId, 'team:create', []);
            $this->tg->answerCallbackQuery($cbId, '');
            $this->tg->sendMessage($chatId, "✍️ ارسل اسم الفريق:");
            return;
        }

        if (str_starts_with($data, 'team:view:')) {
            $teamId = (int)substr($data, strlen('team:view:'));
            if (!$this->teams->isMember($teamId, $userId)) { $this->tg->answerCallbackQuery($cbId, 'ليس عضو'); return; }

            $team = $this->teams->getById($teamId);
            if (!$team) { $this->tg->answerCallbackQuery($cbId, 'غير موجود'); return; }

            $members = $this->teams->listMembers($teamId);
            $text = "👥 <b>" . Helpers::esc($team['name']) . "</b>\n\nالأعضاء:\n";
            foreach ($members as $m) {
                $name = trim(($m['first_name'] ?? '') . ' ' . ($m['last_name'] ?? ''));
                if ($name === '') $name = '@' . ($m['username'] ?? 'user');
                $text .= "• " . Helpers::esc($name) . " <i>({$m['member_role']})</i>\n";
            }
            $link = "https://t.me/" . Config::BOT_USERNAME . "?start=join_" . $team['invite_token'];
            $text .= "\nرابط الدعوة:\n" . Helpers::esc($link);

            $this->tg->answerCallbackQuery($cbId, '');
            $this->tg->sendMessage($chatId, $text, ['reply_markup'=>json_encode(['inline_keyboard'=>[[['text'=>'🔗 رابط الدعوة','url'=>$link]]]], JSON_UNESCAPED_UNICODE)]);
            return;
        }

        $this->tg->answerCallbackQuery($cbId, '');
    }
}
