<?php
declare(strict_types=1);

namespace App\Core;

use App\Controllers\AdminController;
use App\Controllers\MainController;
use App\Controllers\TaskController;
use App\Controllers\TeamController;
use App\Services\AuthService;
use App\Services\ForcedChannelService;

final class Router
{
    private Telegram $tg;
    private AuthService $auth;
    private ForcedChannelService $forced;

    private MainController $main;
    private TaskController $tasks;
    private TeamController $teams;
    private AdminController $admin;

    public function __construct()
    {
        $this->tg = new Telegram();
        $this->auth = new AuthService();
        $this->forced = new ForcedChannelService($this->tg);

        $this->main = new MainController($this->tg, $this->auth);
        $this->tasks = new TaskController($this->tg, $this->auth);
        $this->teams = new TeamController($this->tg, $this->auth);
        $this->admin = new AdminController($this->tg, $this->auth);
    }

    public function handle(array $update): void
    {
        if (isset($update['message'])) { $this->handleMessage($update['message']); return; }
        if (isset($update['callback_query'])) { $this->handleCallback($update['callback_query']); return; }
    }

    private function handleMessage(array $message): void
    {
        $chat = $message['chat'] ?? [];
        $from = $message['from'] ?? [];
        $chatId = (int)($chat['id'] ?? 0);
        $userId = (int)($from['id'] ?? 0);
        $text = Helpers::cleanText((string)($message['text'] ?? ''));
        if (!$chatId || !$userId) return;

        $this->auth->upsertFromTelegram($from, $chat);
        if ($this->auth->isBanned($userId)) return;

        if (!$this->forced->ensureSubscribedOrPrompt($userId, $chatId)) return;

        if (str_starts_with($text, '/start')) {
            $parts = explode(' ', $text, 2);
            $param = $parts[1] ?? '';
            if (str_starts_with($param, 'join_')) {
                $this->teams->handleJoin($chatId, $userId, substr($param, 5));
                return;
            }
            $this->main->start($chatId, $userId);
            return;
        }

        $stateRow = $this->auth->getStateRow($userId);
        if ($stateRow) {
            $state = (string)$stateRow['state'];
            if (str_starts_with($state, 'task:')) { $this->tasks->handleState($chatId, $userId, $text, $stateRow); return; }
            if (str_starts_with($state, 'team:')) { $this->teams->handleState($chatId, $userId, $text, $stateRow); return; }
            if (str_starts_with($state, 'admin:')) { $this->admin->handleState($chatId, $userId, $text, $stateRow); return; }
        }

        switch ($text) {
            case '✅ مهامي': $this->tasks->showMyTasks($chatId, $userId); return;
            case '➕ إضافة مهمة': $this->tasks->askAddPersonal($chatId, $userId); return;
            case '👥 فرقي': $this->teams->showTeams($chatId, $userId); return;
            case 'ℹ️ حسابي': $this->main->account($chatId, $userId); return;
            case '👑 لوحة الأدمن': $this->admin->panel($chatId, $userId); return;
            default: $this->main->help($chatId, $userId); return;
        }
    }

    private function handleCallback(array $cb): void
    {
        $cbId = (string)($cb['id'] ?? '');
        $from = $cb['from'] ?? [];
        $msg = $cb['message'] ?? [];
        $chat = $msg['chat'] ?? [];
        $chatId = (int)($chat['id'] ?? 0);
        $userId = (int)($from['id'] ?? 0);
        $data = (string)($cb['data'] ?? '');
        if (!$cbId || !$chatId || !$userId) return;

        $this->auth->upsertFromTelegram($from, $chat);
        if ($this->auth->isBanned($userId)) return;

        if ($data === 'force:check') {
            $ok = $this->forced->isUserSubscribedToAll($userId);
            $this->tg->answerCallbackQuery($cbId, $ok ? '✅ تم التحقق' : 'اشترك ثم أعد المحاولة');
            if ($ok) $this->main->start($chatId, $userId);
            return;
        }

        if (!$this->forced->ensureSubscribedOrPrompt($userId, $chatId)) {
            $this->tg->answerCallbackQuery($cbId, 'اشترك أولاً');
            return;
        }

        if (str_starts_with($data,'task:')) { $this->tasks->handleCallback($chatId, $userId, $cbId, $data, $msg); return; }
        if (str_starts_with($data,'team:')) { $this->teams->handleCallback($chatId, $userId, $cbId, $data, $msg); return; }
        if (str_starts_with($data,'admin:')) { $this->admin->handleCallback($chatId, $userId, $cbId, $data); return; }

        $this->tg->answerCallbackQuery($cbId, 'غير معروف');
    }
}
