<?php
declare(strict_types=1);

namespace App\Core;

final class Telegram
{
    private string $base;

    public function __construct()
    {
        $this->base = 'https://api.telegram.org/bot' . Config::BOT_TOKEN . '/';
    }

    private function req(string $method, array $data = []): array
    {
        $ch = curl_init($this->base . $method);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $data,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_TIMEOUT => 20,
        ]);
        $out = curl_exec($ch);
        $err = curl_error($ch);
        curl_close($ch);

        if ($out === false) {
            Logger::error('Telegram curl error', ['err' => $err]);
            return ['ok' => false, 'error' => $err];
        }
        $json = json_decode($out, true);
        return is_array($json) ? $json : ['ok' => false, 'raw' => $out];
    }

    public function sendMessage(int $chatId, string $text, array $opts = []): array
    {
        $data = array_merge([
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => true,
        ], $opts);
        return $this->req('sendMessage', $data);
    }

    public function answerCallbackQuery(string $id, string $text = ''): array
    {
        $data = ['callback_query_id' => $id];
        if ($text !== '') $data['text'] = $text;
        return $this->req('answerCallbackQuery', $data);
    }

    public function getChatMember(string $channel, int $userId): array
    {
        return $this->req('getChatMember', ['chat_id' => $channel, 'user_id' => $userId]);
    }
}
