<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Core\DB;
use App\Core\Helpers;

final class TaskRepo
{
    public function createPersonal(int $ownerUserId, string $title): int
    {
        $pdo = DB::pdo();
        $sql = "INSERT INTO tasks (team_id, owner_user_id, assigned_user_id, title, status, created_at, updated_at)
                VALUES (NULL,:o,NULL,:t,'NEW',:c,:u)";
        $pdo->prepare($sql)->execute([':o'=>$ownerUserId, ':t'=>$title, ':c'=>Helpers::now(), ':u'=>Helpers::now()]);
        return (int)$pdo->lastInsertId();
    }

    public function createTeamTask(int $teamId, int $ownerUserId, ?int $assignedUserId, string $title): int
    {
        $pdo = DB::pdo();
        $sql = "INSERT INTO tasks (team_id, owner_user_id, assigned_user_id, title, status, created_at, updated_at)
                VALUES (:team_id,:owner_user_id,:assigned_user_id,:title,'NEW',:created_at,:updated_at)";
        $pdo->prepare($sql)->execute([
            ':team_id'=>$teamId,
            ':owner_user_id'=>$ownerUserId,
            ':assigned_user_id'=>$assignedUserId,
            ':title'=>$title,
            ':created_at'=>Helpers::now(),
            ':updated_at'=>Helpers::now(),
        ]);
        return (int)$pdo->lastInsertId();
    }

    public function listPersonal(int $userId, int $limit=20): array
    {
        $st = DB::pdo()->prepare("SELECT * FROM tasks WHERE team_id IS NULL AND owner_user_id=:uid ORDER BY id DESC LIMIT {$limit}");
        $st->execute([':uid'=>$userId]);
        return $st->fetchAll();
    }

    public function listAssignedToUser(int $userId, int $limit=20): array
    {
        $sql = "SELECT t.*, tm.name AS team_name
                FROM tasks t
                LEFT JOIN teams tm ON tm.id=t.team_id
                WHERE t.team_id IS NOT NULL AND t.assigned_user_id=:uid
                ORDER BY t.id DESC LIMIT {$limit}";
        $st = DB::pdo()->prepare($sql);
        $st->execute([':uid'=>$userId]);
        return $st->fetchAll();
    }

    public function getById(int $taskId): ?array
    {
        $st = DB::pdo()->prepare("SELECT * FROM tasks WHERE id=:id LIMIT 1");
        $st->execute([':id'=>$taskId]);
        $r = $st->fetch();
        return $r ?: null;
    }

    public function updateStatus(int $taskId, string $newStatus): void
    {
        DB::pdo()->prepare("UPDATE tasks SET status=:s, updated_at=:u WHERE id=:id")
            ->execute([':s'=>$newStatus, ':u'=>Helpers::now(), ':id'=>$taskId]);
    }
}
