<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Core\DB;
use App\Core\Helpers;

final class TeamRepo
{
    public function create(int $ownerUserId, string $name, string $inviteToken): int
    {
        $pdo = DB::pdo();
        $pdo->prepare("INSERT INTO teams (owner_user_id, name, invite_token, created_at) VALUES (:o,:n,:t,:c)")
            ->execute([':o'=>$ownerUserId, ':n'=>$name, ':t'=>$inviteToken, ':c'=>Helpers::now()]);
        return (int)$pdo->lastInsertId();
    }

    public function getById(int $teamId): ?array
    {
        $st = DB::pdo()->prepare("SELECT * FROM teams WHERE id=:id LIMIT 1");
        $st->execute([':id'=>$teamId]);
        $r = $st->fetch();
        return $r ?: null;
    }

    public function getByInviteToken(string $token): ?array
    {
        $st = DB::pdo()->prepare("SELECT * FROM teams WHERE invite_token=:t LIMIT 1");
        $st->execute([':t'=>$token]);
        $r = $st->fetch();
        return $r ?: null;
    }

    public function listForUser(int $userId): array
    {
        $sql = "SELECT t.* FROM teams t
                JOIN team_members m ON m.team_id=t.id
                WHERE m.user_id=:uid
                ORDER BY t.id DESC";
        $st = DB::pdo()->prepare($sql);
        $st->execute([':uid'=>$userId]);
        return $st->fetchAll();
    }

    public function addMember(int $teamId, int $userId, string $role='MEMBER'): void
    {
        $sql = "INSERT INTO team_members (team_id, user_id, member_role, joined_at)
                VALUES (:tid,:uid,:r,:t)
                ON DUPLICATE KEY UPDATE member_role=member_role";
        DB::pdo()->prepare($sql)->execute([':tid'=>$teamId, ':uid'=>$userId, ':r'=>$role, ':t'=>Helpers::now()]);
    }

    public function listMembers(int $teamId): array
    {
        $sql = "SELECT u.user_id, u.chat_id, u.username, u.first_name, u.last_name, m.member_role
                FROM team_members m JOIN users u ON u.user_id=m.user_id
                WHERE m.team_id=:tid
                ORDER BY (m.member_role='OWNER') DESC, u.id ASC";
        $st = DB::pdo()->prepare($sql);
        $st->execute([':tid'=>$teamId]);
        return $st->fetchAll();
    }

    public function isMember(int $teamId, int $userId): bool
    {
        $st = DB::pdo()->prepare("SELECT 1 FROM team_members WHERE team_id=:tid AND user_id=:uid LIMIT 1");
        $st->execute([':tid'=>$teamId, ':uid'=>$userId]);
        return (bool)$st->fetchColumn();
    }

    public function isOwner(int $teamId, int $userId): bool
    {
        $st = DB::pdo()->prepare("SELECT 1 FROM team_members WHERE team_id=:tid AND user_id=:uid AND member_role='OWNER' LIMIT 1");
        $st->execute([':tid'=>$teamId, ':uid'=>$userId]);
        return (bool)$st->fetchColumn();
    }
}
