<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Core\DB;
use App\Core\Helpers;
use App\Core\Telegram;

final class UserRepo
{
    public function upsert(int $chatId, int $userId, ?string $username, ?string $first, ?string $last, string $role): void
    {
        $pdo = DB::pdo();
        $sql = "INSERT INTO users (chat_id, user_id, username, first_name, last_name, role, created_at, last_seen_at)
                VALUES (:chat_id,:user_id,:username,:first_name,:last_name,:role,:created_at,:last_seen_at)
                ON DUPLICATE KEY UPDATE
                    chat_id=VALUES(chat_id),
                    username=VALUES(username),
                    first_name=VALUES(first_name),
                    last_name=VALUES(last_name),
                    last_seen_at=VALUES(last_seen_at)";
        $pdo->prepare($sql)->execute([
            ':chat_id'=>$chatId,
            ':user_id'=>$userId,
            ':username'=>$username,
            ':first_name'=>$first,
            ':last_name'=>$last,
            ':role'=>$role,
            ':created_at'=>Helpers::now(),
            ':last_seen_at'=>Helpers::now(),
        ]);
    }

    public function getByUserId(int $userId): ?array
    {
        $st = DB::pdo()->prepare("SELECT * FROM users WHERE user_id=:uid LIMIT 1");
        $st->execute([':uid'=>$userId]);
        $r = $st->fetch();
        return $r ?: null;
    }

    public function setRoleByChatId(int $chatId, string $role): bool
    {
        $st = DB::pdo()->prepare("UPDATE users SET role=:r WHERE chat_id=:cid");
        $st->execute([':r'=>$role, ':cid'=>$chatId]);
        return $st->rowCount() > 0;
    }

    public function banByChatId(int $chatId): bool
    {
        $st = DB::pdo()->prepare("UPDATE users SET is_banned=1 WHERE chat_id=:cid");
        $st->execute([':cid'=>$chatId]);
        return $st->rowCount() > 0;
    }

    public function listAllActive(): array
    {
        return DB::pdo()->query("SELECT * FROM users WHERE is_banned=0 ORDER BY id ASC")->fetchAll();
    }

    public function countUsers(): int { return (int)DB::pdo()->query("SELECT COUNT(*) c FROM users")->fetch()['c']; }
    public function countAdmins(): int { return (int)DB::pdo()->query("SELECT COUNT(*) c FROM users WHERE role='ADMIN'")->fetch()['c']; }
    public function countBanned(): int { return (int)DB::pdo()->query("SELECT COUNT(*) c FROM users WHERE is_banned=1")->fetch()['c']; }

    public function broadcast(Telegram $tg, string $text, int $sleepMs = 50): int
    {
        $users = $this->listAllActive();
        $sent = 0;
        foreach ($users as $u) {
            $tg->sendMessage((int)$u['chat_id'], $text);
            $sent++;
            if ($sleepMs > 0) usleep($sleepMs * 1000);
        }
        return $sent;
    }
}
