<?php
declare(strict_types=1);

namespace App\Services;

use App\Core\Config;
use App\Core\DB;
use App\Core\Helpers;
use App\Repositories\UserRepo;

final class AuthService
{
    private UserRepo $users;

    public function __construct()
    {
        $this->users = new UserRepo();
    }

    public function upsertFromTelegram(array $from, array $chat): void
    {
        $chatId = (int)($chat['id'] ?? 0);
        $userId = (int)($from['id'] ?? 0);
        if (!$chatId || !$userId) return;

        $username = $from['username'] ?? null;
        $first = $from['first_name'] ?? null;
        $last = $from['last_name'] ?? null;

        $forceAdmin = ($chatId === Config::SUPER_ADMIN_CHAT_ID);
        $role = $forceAdmin ? 'ADMIN' : 'USER';

        $this->users->upsert($chatId, $userId, $username, $first, $last, $role);

        if ($forceAdmin) {
            DB::pdo()->prepare("UPDATE users SET role='ADMIN' WHERE user_id=:uid")->execute([':uid'=>$userId]);
        }
    }

    public function getUser(int $userId): ?array { return $this->users->getByUserId($userId); }
    public function isAdmin(int $userId): bool { $u=$this->getUser($userId); return $u && $u['role']==='ADMIN'; }
    public function isBanned(int $userId): bool { $u=$this->getUser($userId); return $u && (int)$u['is_banned']===1; }

    public function getStateRow(int $userId): ?array
    {
        $st = DB::pdo()->prepare("SELECT user_id, state, payload, updated_at FROM user_states WHERE user_id=:uid LIMIT 1");
        $st->execute([':uid'=>$userId]);
        $r = $st->fetch();
        if (!$r) return null;
        $payload = [];
        if (!empty($r['payload'])) {
            $decoded = json_decode((string)$r['payload'], true);
            if (is_array($decoded)) $payload = $decoded;
        }
        $r['payload'] = $payload;
        return $r;
    }

    public function setState(int $userId, string $state, array $payload=[]): void
    {
        $sql = "INSERT INTO user_states (user_id, state, payload, updated_at)
                VALUES (:uid,:st,:pl,:u)
                ON DUPLICATE KEY UPDATE state=VALUES(state), payload=VALUES(payload), updated_at=VALUES(updated_at)";
        DB::pdo()->prepare($sql)->execute([
            ':uid'=>$userId,
            ':st'=>$state,
            ':pl'=>json_encode($payload, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
            ':u'=>Helpers::now(),
        ]);
    }

    public function clearState(int $userId): void
    {
        DB::pdo()->prepare("DELETE FROM user_states WHERE user_id=:uid")->execute([':uid'=>$userId]);
    }
}
