<?php
declare(strict_types=1);

namespace App\Services;

use App\Repositories\ForcedChannelRepo;
use App\Core\Telegram;

final class ForcedChannelService
{
    private ForcedChannelRepo $repo;
    private Telegram $tg;

    public function __construct(Telegram $tg)
    {
        $this->repo = new ForcedChannelRepo();
        $this->tg = $tg;
    }

    public function getActiveChannels(): array { return $this->repo->listActive(); }

    public function isUserSubscribedToAll(int $userId): bool
    {
        $chs = $this->getActiveChannels();
        if (!$chs) return true;

        foreach ($chs as $c) {
            $username = (string)$c['channel_username'];
            $res = $this->tg->getChatMember($username, $userId);
            if (!($res['ok'] ?? false)) return false;
            $status = $res['result']['status'] ?? '';
            if (in_array($status, ['left', 'kicked'], true) || $status === '') return false;
        }
        return true;
    }

    public function ensureSubscribedOrPrompt(int $userId, int $chatId): bool
    {
        $chs = $this->getActiveChannels();
        if (!$chs) return true;
        if ($this->isUserSubscribedToAll($userId)) return true;

        $text = "🔒 <b>الاشتراك الإجباري</b>\n\nاشترك في القنوات التالية ثم اضغط <b>تحقق</b>:\n\n";
        $kb = [];
        foreach ($chs as $c) {
            $u = (string)$c['channel_username'];
            $link = (string)$c['invite_url'];
            $text .= "• {$u}\n";
            $kb[] = [[ 'text' => '📌 ' . $u, 'url' => $link ]];
        }
        $kb[] = [[ 'text' => '✅ تحقق', 'callback_data' => 'force:check' ]];

        $this->tg->sendMessage($chatId, $text, [
            'reply_markup' => json_encode(['inline_keyboard'=>$kb], JSON_UNESCAPED_UNICODE)
        ]);
        return false;
    }
}
